# Code for 'Boosting Randomized Smoothing with Variance Reduced Classifiers'

## Overview

The following files are necessary to load the datasets and networks required to reproduce our experiments:

- `datasets.py`: needed to get the datasets (`CIFAR10` and `ImageNet`)
- `architectures.py`: needed to get the architectures (`ResNet20` and `ResNet110` for `CIFAR10` and `ResNet50` for `ImageNet`)
- `archs/cifar_resnet.py`: needed for the `CIFAR10` architectures
- `archs/dncnn.py`: needed for denoiser models

The following files are the core randomized smoothing files:

- `smooth_ensemble.py`: core randomized smoothing class which supports ensembles as base classifiers
- `smooth_ensemble_k_consensus.py`: randomized smoothing class for ensembles which supports k-consensus early stopping
- `smooth_ensemble_adaptive.py`: does randomized smoothing via adaptive sampling for single models or ensembles
- `smooth_ensemble_adaptive_and_k_consensus.py`: does randomized smoothing via adaptive sampling and k-consensus for ensembles
- `smooth_ensemble_denoising.py`: supports an ensemble of denoised classifiers for randomized smoothing

The following files can be used to do the certifications via the corresponding randomized smoothing methods above:
- `certify_ensemble.py`
- `certify_ensemble_k_consensus.py`
- `certify_ensemble_adaptive.py`
- `certify_ensemble_adaptive_and_k_consensus.py`
- `certify_ensemble_denoising.py`

There are also files which process the outputted log files into data used in the paper:
- `analyze_ensemble.py`: contains a function which creates a latex table for files which were generated by `certify_ensemble.py`, `certify_ensemble_k_consensus.py` or by `certify_ensemble_denoising.py`.
- `analyze_adaptive_sampling.py`: contains a function which creates a latex table fir files which were generatey by `certify_ensemble_adaptive.py` or by `certify_ensemble_adaptive_and_k_consensus.py`

Concrete scripts to reproduce our experiments can be found in the folder `scripts`.

## Getting Started

To run things, first please install the required environment:

```
conda create --name smooth_ensembles python=3.8
conda activate smooth_ensembles
pip install -r requirements.txt
```

To install pytorch 1.8.0 and torchvision 0.9.0 you can use the following command (depending on your installed CUDA version (can be checked e.g., by running `nvidia-smi`)):
```
# CUDA 10.2
conda install pytorch==1.8.0 torchvision==0.9.0 torchaudio==0.8.0 cudatoolkit=10.2 -c pytorch

# CUDA 11.1
conda install pytorch==1.8.0 torchvision==0.9.0 torchaudio==0.8.0 cudatoolkit=11.1 -c pytorch -c conda-forge
```
If you are not using the most recent GPU drivers, please see here (https://docs.nvidia.com/deploy/cuda-compatibility/index.html) for a compatible cudatoolkit version.

Models can be trained using the code published along the corresponding training methods and by following their instructions. 
Namely, https://github.com/locuslab/smoothing for Gaussian, https://github.com/jh-jeong/smoothing-consistency for Consistency and https://github.com/Hadisalman/smoothing-adversarial for SmoothAdv. 
The exact parameters and commands we use are detailed in `training.md`.

Alternatively, we provide the trained models we use here: https://mega.nz/file/zeQlVQAS#p48AffIzrl8lOHki5r6P4UwAy3GZt-1TmxKO0q5UHOI. 
Please make sure to move the folders `models-cifar10` and `models-imagenet` into the root directory of this code base to run the scripts.
This can be done by executing the following commands after downloading the models to your `~/Downloads` directory:
```
tar -xf ~/Downloads/models_smoothing_ensembles.tar.gz -C ./
mv models_smoothing_ensembles/* ./.
rm models_smoothing_ensembles/ -r
```

## Examples

Here are some examples of how to use this code base. The folder `scripts` contains a lot more detailed examples.

### Example - Ensembles

Here is a basic example of certifying robustness via `certify_ensemble.py` for a base classifier which sonsists of 3 single models:

```
CUDA_VISIBLE_DEVICES=0 python certify_ensemble.py \
    cifar10 \
    0.25 \
    ./output_dir/output_file \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1000.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1001.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1002.pth.tar \
    --alpha 0.001 \
    --N 100000 \
    --skip 20 \
    --batch 1000
```

The first argument is the dataset (here `cifar10`), the second is the noise with which we do the certification (here `sigma = 0.25`) and the third is the path to the output files (here `output_dir/output_file`). Note that the `output_file_i` will save the output file for the `i/2`-th single model for even `i`, and the output file of the ensemble of the first `(i+1)/2` single models for odd `i`. The following arguments are the paths to the models, we can have as many models as we want after each other. The remaning arguments are optional.

### Example - Ensembles via K-Consensus

In addition to ensembles, here we also apply k-consensus, i.e. if the first k models agree on the class of a perturbed sample, we stop. 
The command is the same as for standard ensembles with with the exception of the additional argument `--voting_size` which determines k and the fact that `output_file` only saves the output of the whole ensemble, i.e. we don't save results of each individual model and smaller ensembles.

```
CUDA_VISIBLE_DEVICES=0 python certify_ensemble_k_consensus.py \
    cifar10 \
    0.25 \
    ./output_dir/output_file \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1000.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1001.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1002.pth.tar \
    --alpha 0.001 \
    --N 100000 \
    --skip 20 \
    --batch 1000 \
    --voting_size 2
```

### Example - Adaptive Sampling

Now we provide an example of randomized smoothing via adaptive sampling using `certify_ensemble_adaptive.py`.\
The key interface change compared to `certify_ensemble.py` is that now we have the additional argument `radius_to_certify` which is the predetermined radius we want to consider for certification and the numbers `Ni` (for currently `i` from `{1, 2, 3, 4, 5}`) determine how many samples should be used in the `i`-th step.

```
CUDA_VISIBLE_DEVICES=0 python certify_ensemble_adaptive.py \
    cifar10 \
    0.25 \
    ./output_dir/output_file \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1000.pth.tar \
    --alpha 0.001 \
    --N0 100 \
    --skip 20 \
    --batch 1000 \
    --beta 0.0001 \
    --radius_to_certify 0.25 \
    --N1 1000 \
    --N2 10000 \
    --N3 125000 
```

Note that similarly to above, we can list an arbitrary number of models if we want to consider an ensemble as a base classifier.


### Example - Adaptive Sampling and K-Consensus

The file `certify_ensemble_adaptive_and_k_consensus.py` combines all the previous ideas and supports randomized smoothing via ensembles, and certification time reduction via `k`-consensus and adaptive sampling. 
The arguments are essentially the union of the arguments from `certify_adaptive_k_consensus.py` and `certify_ensemble_adaptive.py`.

```
CUDA_VISIBLE_DEVICES=0 python certify_ensemble_adaptive_and_k_consensus.py \
    cifar10 \
    0.25 \
    ./output_dir/output_file \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1000.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1001.pth.tar \
    ./models-cifar10/consistency/resnet110/0.25/checkpoint-1002.pth.tar \
    --alpha 0.001 \
    --N0 100 \
    --N 100000 \
    --skip 20 \
    --batch 1000 \
    --beta 0.0001 \
    --voting_size 2 \
    --radius_to_certify 0.25 \
    --N1 1000 \
    --N2 10000 \
    --N3 125000
```

### Example - Ensembles for Denoised Smoothing

Here, we give an example of ensembles for denoised smoothing via `certify_ensemble_denoising.py`. The key difference to `certify_ensemble.py` is that here, we first have to provide the classifier trained on unperturbed samples. Then we list the denoisers.

```
CUDA_VISIBLE_DEVICES=0 python certify_ensemble_denoising.py \
    cifar10 \
    0.25 \
    ./output_dir/output_file \
    ./models-cifar10/denoised-smoothing/checkpoint-ResNet110_90epochs-noise_0.00.pth.tar \
    ./models-cifar10/denoised-smoothing/checkpoint-stab_obj-cifar10_smoothness_obj_adamThenSgd_3-resnet110_90epochs-dncnn_wide-noise_0.25.pth.tar \
    ./models-cifar10/denoised-smoothing/checkpoint-stab_obj-cifar10_smoothness_obj_adamThenSgd_4-resnet110_90epochs-dncnn_wide-noise_0.25.pth.tar \
    ./models-cifar10/denoised-smoothing/checkpoint-stab_obj-cifar10_smoothness_obj_adamThenSgd_5-resnet110_90epochs-dncnn_wide-noise_0.25.pth.tar \
    ./models-cifar10/denoised-smoothing/checkpoint-stab_obj-cifar10_smoothness_obj_adamThenSgd_1-resnet110_90epochs-dncnn_wide-noise_0.25.pth.tar \
    --alpha 0.001 \
    --N 100000 \
    --skip 20 \
    --batch 1000
```



